#include "viewport.h"

#include <math.h>
#include <stdlib.h>

#include "bitmap.h"
#include "level.h"

#define MIN(a, b)          ((a) < (b) ? (a) : (b))
#define MAX(a, b)          ((a) > (b) ? (a) : (b))
#define CLAMP(x, min, max) MAX(MIN(x, max), min)

struct ScreenRect {
	float x0, x1;
	int   xp0, xp1;
	float y00, y01, y10, y11;
};

static int GetScreenRect(struct Camera *, int, int, int, int, int, int,
						 struct ScreenRect *, const struct ScreenRect *);

static void FillScreenRect(struct Camera *, int, int, int, int, int, int,
					 const struct ScreenRect *, int);
static void DrawWall(struct Camera *, int, int, int, int, int, int,
					 const struct ScreenRect *);

static void DrawHalfEdge(struct Camera *, struct Level *, const kke_halfedge_t *, const struct ScreenRect *, int);
static void DrawSector(struct Camera *, struct Level *, const kke_sector_t *, const struct ScreenRect *, int);

static struct Bitmap *s_bmp;

/* Cached sine and cosine of the camera's rotation, be sure to update this before drawing. */
static float s_rSin, s_rCos;
static int s_numEdgesRendered;

static const kke_sector_t *s_pCurrentSector;

void View_Render(struct Bitmap *bmp, struct Camera *camera, struct Level *lvl)
{
	struct ScreenRect fullScreen;
	float aspect;
	int i;

	s_bmp = bmp;
	aspect = (float)s_bmp->w / (float)s_bmp->h;

	s_rSin = (float)sin(camera->rot);
	s_rCos = (float)cos(camera->rot);

	if (lvl->header.numSectors == 0)
		return;

	fullScreen.x0 = -aspect;
	fullScreen.x1 =  aspect;
	fullScreen.xp0 = (int)((fullScreen.x0 + aspect) * s_bmp->h / 2.f);
	fullScreen.xp1 = (int)((fullScreen.x1 + aspect) * s_bmp->h / 2.f);
	fullScreen.y00 = -1.f;
	fullScreen.y01 =  1.f;
	fullScreen.y10 = -1.f;
	fullScreen.y11 =  1.f;

	/* Find the sector that the camera is currently in. */
	for (i = 0; i < lvl->header.numSectors; i++) {
		const kke_sector_t *sc = &lvl->sectors[i];
		int inX = (camera->x >= sc->x0 && camera->x < sc->x1);
		int inY = (camera->z >= sc->y0 && camera->z < sc->y1);
		if (inX && inY) {
			s_pCurrentSector = sc;
			camera->y = sc->floorHeight + 1.25f;
			break;
		}
	}

	if (s_pCurrentSector == NULL) {
		s_pCurrentSector = &lvl->sectors[0];
	}

	s_numEdgesRendered = 0;
	DrawSector(camera, lvl, s_pCurrentSector, &fullScreen, 0xff1144dd);
}

static int GetScreenRect(struct Camera *camera, int x0, int y0, int z0, int x1, int y1, int z1,
						 struct ScreenRect *screen, const struct ScreenRect *bounds)
{
	const float aspect = (float)s_bmp->w / (float)s_bmp->h;
	const float nearClip = 0.01f;

	/* World-space coordinates relative to the camera. */
	float xx0 = camera->x - x0;
	float yy0 = camera->y - y0;
	float zz0 = camera->z - z0;
	float xx1 = camera->x - x1;
	float yy1 = camera->y - y1;
	float zz1 = camera->z - z1;

	float xr0 = xx0 * s_rCos - zz0 * s_rSin;
	float zr0 = xx0 * s_rSin + zz0 * s_rCos;

	float xr1 = xx1 * s_rCos - zz1 * s_rSin;
	float zr1 = xx1 * s_rSin + zz1 * s_rCos;

	if (zr0 < nearClip && zr1 < nearClip)
		return 0;

	if (zr0 < nearClip) {
		float p = (nearClip - zr0) / (zr1 - zr0);
		xr0 = xr0 + (xr1 - xr0) * p;
		zr0 = zr0 + (zr1 - zr0) * p;
	}

	if (zr1 < nearClip) {
		float p = (nearClip - zr0) / (zr1 - zr0);
		xr1 = xr0 + (xr1 - xr0) * p;
		zr1 = zr0 + (zr1 - zr0) * p;
	}

	screen->x0  = xr0 / zr0;
	screen->x1  = xr1 / zr1;
	screen->y00 = yy0 / zr0;
	screen->y01 = yy1 / zr0;
	screen->y10 = yy0 / zr1;
	screen->y11 = yy1 / zr1;

	screen->xp0 = (int)((screen->x0 + aspect) * s_bmp->h / 2.f);
	screen->xp1 = (int)((screen->x1 + aspect) * s_bmp->h / 2.f);

	if (screen->xp0 < bounds->xp0) screen->xp0 = bounds->xp0;
	if (screen->xp1 > bounds->xp1) screen->xp1 = bounds->xp1;

	/* We are viewing from behind. */
	if (screen->xp1 <= screen->xp0)
		return 0;

	/* Outside the given bounds. */
	if (screen->xp0 >= bounds->xp1 || screen->xp1 <= bounds->xp0)
		return 0;

	return 1;
}

static void FillScreenRect(struct Camera *camera, int x0, int y0, int z0, int x1, int y1, int z1,
						   const struct ScreenRect *bounds, int col)
{
	struct ScreenRect screen;
	int x, y;
	float xpf0, xpf1;
	float xpfb0, xpfb1;
	float invLen, invLenBds;

	const float aspect = (float)s_bmp->w / (float)s_bmp->h;
	if (GetScreenRect(camera, x0, y0, z0, x1, y1, z1, &screen, bounds) == 0) {
		return;
	}

	/* Bitmap pixel coordinates (0 to width or height) */
	xpf0 = (screen.x0 + aspect) * s_bmp->h / 2.f;
	xpf1 = (screen.x1 + aspect) * s_bmp->h / 2.f;

	xpfb0 = (bounds->x0 + aspect) * s_bmp->h / 2.f;
	xpfb1 = (bounds->x1 + aspect) * s_bmp->h / 2.f;

	invLen    = 1 / (xpf1 - xpf0);
	invLenBds = 1 / (xpfb1 - xpfb0);

	for (x = screen.xp0; x < screen.xp1; x++) {
		float distAcross    = (x - xpf0) * invLen;
		float distAcrossBds = (x - xpfb0) * invLenBds;

		float ys0 = screen.y00 + (screen.y10 - screen.y00) * distAcross;
		float ys1 = screen.y01 + (screen.y11 - screen.y01) * distAcross;

		int yp0 = (int)((ys0 + 1) * s_bmp->h / 2.f);
		int yp1 = (int)((ys1 + 1) * s_bmp->h / 2.f);

		float ysb0 = bounds->y00 + (bounds->y10 - bounds->y00) * distAcrossBds;
		float ysb1 = bounds->y01 + (bounds->y11 - bounds->y01) * distAcrossBds;

		int ypb0 = (int)((ysb0 + 1) * s_bmp->h / 2.f);
		int ypb1 = (int)((ysb1 + 1) * s_bmp->h / 2.f);

		if (yp0 < 0)         yp0 = 0;
		if (yp1 >= s_bmp->h) yp1 = s_bmp->h;

		if (ypb0 < 0)         ypb0 = 0;
		if (ypb1 >= s_bmp->h) ypb1 = s_bmp->h;

		if (x < bounds->xp0 || x >= bounds->xp1)
			continue;

		for (y = yp0; y < yp1; y++) {
			if (y < ypb0 || y >= ypb1)
				continue;
			s_bmp->pixels[x + y * s_bmp->w] = col;
		}
	}
}


static void DrawWall(struct Camera *camera, int x0, int y0, int z0, int x1, int y1, int z1,
					 const struct ScreenRect *bounds)
{
	FillScreenRect(camera, x0, y0, z0, x1, y1, z1, bounds, 0xff00ff00);
}

static void DrawHalfEdge(struct Camera *camera, struct Level *lvl,
						 const kke_halfedge_t *edge, const struct ScreenRect *bounds, int col)
{
	struct ScreenRect edgeRect;
	int xw0, yw0, zw0, xw1, yw1, zw1;

	const float aspect = (float)s_bmp->w / (float)s_bmp->h;
	
	const kke_halfedge_t *otherEdge = &lvl->edges[edge->otherHalf];

	const kke_sector_t *sector      = &lvl->sectors[edge->sector];
	const kke_sector_t *otherSector = &lvl->sectors[otherEdge->sector];

	if (otherSector->floorHeight >= sector->ceilHeight
		|| otherSector->ceilHeight <= sector->floorHeight)
		return;

	yw0 = CLAMP(otherSector->ceilHeight, sector->floorHeight, sector->ceilHeight);
	yw1 = CLAMP(otherSector->floorHeight, sector->floorHeight, sector->ceilHeight);

	if (otherSector->y1 == sector->y0) {
		/* Edge is along northern wall. */
		xw0 = sector->x0 + edge->x + edge->length;
		zw0 = sector->y0;
		xw1 = sector->x0 + edge->x;
		zw1 = sector->y0;
	} else if (otherSector->x1 == sector->x0) {
		/* Edge is along western wall. */
		xw0 = sector->x0;
		zw0 = sector->y0 + edge->y;
		xw1 = sector->x0;
		zw1 = sector->y0 + edge->y + edge->length;
	} else if (otherSector->x0 == sector->x1) {
		/* Edge is along eastern wall. */
		xw0 = sector->x1;
		zw0 = sector->y0 + edge->y + edge->length;
		xw1 = sector->x1;
		zw1 = sector->y0 + edge->y;
	} else if (otherSector->y0 == sector->y1) {
		/* Edge is along southern wall. */
		xw0 = sector->x0 + edge->x;
		zw0 = sector->y1;
		xw1 = sector->x0 + edge->x + edge->length;
		zw1 = sector->y1;
	} else {
		/* Edge doesn't seem to be along a wall. */
		return;
	}

	if (GetScreenRect(camera, xw0, yw0, zw0, xw1, yw1, zw1, &edgeRect, bounds) == 1) {
		unsigned r, g, b;
		int newCol;

		r = (col >> 16) & 0xff;
		g = (col >>  8) & 0xff;
		b = (col >>  0) & 0xff;

		if (r >= 16) r -= 16;
		if (g >= 16) g -= 16;
		if (b >= 16) b -= 16;

		newCol = (r & 0xff) << 16 | (g & 0xff) << 8 | (b & 0xff);

		++s_numEdgesRendered;
		FillScreenRect(camera, xw0, yw0, zw0, xw1, yw1, zw1, bounds, 0xff000000);
		DrawSector(camera, lvl, otherSector, &edgeRect, newCol);
	}
}

static void DrawSector(struct Camera *camera, struct Level *lvl,
					   const kke_sector_t *sc, const struct ScreenRect *bounds, int col)
{
	int i;

	int colDarken = col & 0xffd0d0d0;

	FillScreenRect(camera, sc->x1, sc->ceilHeight, sc->y0, sc->x0, sc->floorHeight, sc->y0, bounds, col);
	FillScreenRect(camera, sc->x1, sc->ceilHeight, sc->y1, sc->x1, sc->floorHeight, sc->y0, bounds, colDarken);
	FillScreenRect(camera, sc->x0, sc->ceilHeight, sc->y1, sc->x1, sc->floorHeight, sc->y1, bounds, col);
	FillScreenRect(camera, sc->x0, sc->ceilHeight, sc->y0, sc->x0, sc->floorHeight, sc->y1, bounds, colDarken);

	if (s_numEdgesRendered > 256) {
		return;
	}

	/* TODO: This is lazy and naive, we should build lists per sector on which
	         edges it owns when we load the level. */
	for (i = 0; i < lvl->header.numHalfEdges; i++) {
		/* Draw edges belonging to this sector. */
		const kke_halfedge_t *edge = &lvl->edges[i];
		if (&lvl->sectors[edge->sector] == sc) {
			DrawHalfEdge(camera, lvl, &lvl->edges[i], bounds, col);
		}
	}
}
