#include "mapdoc.h"

#include <stdio.h>
#include <math.h>
#include <memory.h>
#include "stdafx.h"

#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define MAX(a, b) ((a) > (b) ? (a) : (b))

struct TmpEdge {
	u16 x0, y0;
	u16 x1, y1;
	u16 sectorA, sectorB;
};

struct MapDocument Map_Create()
{
	struct MapDocument doc;
	kke_header_t header;

	header.version = KKE_MAP_VERSION;
	header.startX = 0;
	header.startY = 0;
	header.numSectors = 0;
	header.numHalfEdges = 0;

	doc.header = header;
	doc.szFileName = NULL;
	doc.bNeedsSave = 0;

	return doc;
}

kke_sector_t *Map_AddSector(struct MapDocument *prMap, kke_sector_t newSector)
{
	int idx = prMap->header.numSectors;
	prMap->sectors[idx] = newSector;
	++prMap->header.numSectors;
	prMap->bNeedsSave = 1;
	return &prMap->sectors[idx];
}

void Map_SetFileName(struct MapDocument *map, const char *name)
{
	int namelen = strlen(name);
	if (map->szFileName != NULL)
		free(map->szFileName);
	map->szFileName = malloc(namelen + 1);
	strncpy(map->szFileName, name, namelen + 1);
}

void Map_CalculateEdges(struct MapDocument *prMap)
{
	struct TmpEdge fullEdges[KKE_MAX_HALFEDGES];
	BOOL visited[KKE_MAX_HALFEDGES];
	int edgeIdx = 0, halfEdgeIdx = 0;
	int i, j;

	for (i = 0; i < prMap->header.numSectors; i++) {
		visited[i] = FALSE;
	}

	/* Iterate through each sector, and find every FULL edge between them. */
	for (i = 0; i < prMap->header.numSectors; i++) {
		const kke_sector_t *scA = &prMap->sectors[i];

		for (j = 0; j < prMap->header.numSectors; j++) {
			BOOL north, east, south, west;
			BOOL xIn, yIn;
			struct TmpEdge edge;

			const kke_sector_t *scB = &prMap->sectors[j];

			if (i == j || visited[j] == TRUE)
				continue;

			/* Determine whether these sectors have an overlapping edge. */
			/* NOTE: Given two sectors, it's only possible to have zero or one edge. */

			/* TODO: No point in making an edge if the floors and ceilings are not within range. */

			xIn = (scB->x0 >= scA->x0 && scB->x0 < scA->x1) || (scB->x1 > scA->x0 && scB->x1 <= scA->x1);
			yIn = (scB->y0 >= scA->y0 && scB->y0 < scA->y1) || (scB->y1 > scA->y0 && scB->y1 <= scA->y1);

			north = scA->y0 == scB->y1 && xIn;
			east  = scA->x1 == scB->x0 && yIn;
			south = scA->y1 == scB->y0 && xIn;
			west  = scA->x0 == scB->x1 && yIn;
			if (north) {
				edge.x0 = MAX(scA->x0, scB->x0);
				edge.y0 = scA->y0;
				edge.x1 = MIN(scB->x1, scA->x1);
				edge.y1 = scA->y0;
			} else if (east) {
				edge.x0 = scA->x1;
				edge.y0 = MAX(scA->y0, scB->y0);
				edge.x1 = scA->x1;
				edge.y1 = MIN(scB->y1, scA->y1);
			} else if (south) {
				edge.x0 = MAX(scA->x0, scB->x0);
				edge.y0 = scA->y1;
				edge.x1 = MIN(scB->x1, scA->x1);
				edge.y1 = scA->y1;
			} else if (west) {
				edge.x0 = scA->x0;
				edge.y0 = MAX(scA->y0, scB->y0);
				edge.x1 = scA->x0;
				edge.y1 = MIN(scB->y1, scA->y1);
			}

			if (north || south || east || west) {
				edge.sectorA = i;
				edge.sectorB = j;
				fullEdges[edgeIdx++] = edge;
				visited[i] = TRUE;
			}
		}
		//visited[i] = TRUE;
	}
	
	/* Once we've found all the edges, split them into half-edges. */
	for (i = 0; i < edgeIdx; i++) {
		kke_halfedge_t edgeA, edgeB;
		kke_sector_t *pSectorA, *pSectorB;
		int xLen, yLen, len;

		struct TmpEdge *pEdge = &fullEdges[i];

		pSectorA = &prMap->sectors[pEdge->sectorA];
		pSectorB = &prMap->sectors[pEdge->sectorB];

		xLen = (pEdge->x1 - pEdge->x0);
		yLen = (pEdge->y1 - pEdge->y0);
		len = (int)sqrt(xLen*xLen + yLen*yLen);

		edgeA.x         = pEdge->x0 - pSectorA->x0;
		edgeA.y         = pEdge->y0 - pSectorA->y0;
		edgeA.length    = len;
		edgeA.sector    = pEdge->sectorA;
		edgeA.otherHalf = halfEdgeIdx + 1;

		prMap->edges[halfEdgeIdx++] = edgeA;

		edgeB.x         = pEdge->x0 - pSectorB->x0;
		edgeB.y         = pEdge->y0 - pSectorB->y0;
		edgeB.length    = len;
		edgeB.sector    = pEdge->sectorB;
		edgeB.otherHalf = halfEdgeIdx - 1;

		prMap->edges[halfEdgeIdx++] = edgeB;
	}

	prMap->header.numHalfEdges = halfEdgeIdx;
}

kke_sector_t *Map_SectorAtPoint(const struct MapDocument *pMap, int x, int y)
{
	int i;
	for (i = 0; i < pMap->header.numSectors; i++) {
		kke_sector_t *sc = &pMap->sectors[i];
		if ((x >= sc->x0 && x < sc->x1) && (y >= sc->y0 && y < sc->y1)) {
			return sc;
		}
	}
	return NULL;
}

int Map_IntersectsWithAnother(const struct MapDocument *prMap, const kke_sector_t *scA)
{
	int i;
	for (i = 0; i < prMap->header.numSectors; i++) {
		const kke_sector_t *scB = &prMap->sectors[i];

		/* top left inside square? */
		int xi0 = (scA->x0 >= scB->x0 && scA->x0 < scB->x1);
		int yi0 = (scA->y0 >= scB->y0 && scA->y0 < scB->y1);

		/* bottom right inside square? */
		int xi1 = (scA->x1 > scB->x0 && scA->x1 <= scB->x1);
		int yi1 = (scA->y1 > scB->y0 && scA->y1 <= scB->y1);

		/* Don't check against itself. */
		if (scA == scB)
			continue;

		if (xi0 && yi0 || xi1 && yi1) {
			return 1;
		}
	}
	return 0;
}

int Map_ReadFromFile(struct MapDocument *poMap, const char *szFilename)
{
	HANDLE hFile;
	char *dataBuf;
	DWORD dwBytesRead;
	kke_header_t header;
	int i, offs, rc = 0;

	hFile = CreateFile(szFilename, GENERIC_READ, FILE_SHARE_READ, NULL,
		OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

	if (hFile == INVALID_HANDLE_VALUE)
		return -1;

	dataBuf = malloc(2048);
	if (ReadFile(hFile, dataBuf, 2048, &dwBytesRead, NULL) == FALSE) {
		rc = -1;
		goto cleanup;
	}

	header.version      = *dataBuf;
	header.startX       = *(u16*)(dataBuf + 1);
	header.startY       = *(u16*)(dataBuf + 3);
	header.numSectors   = *(u16*)(dataBuf + 5);
	header.numHalfEdges = *(u16*)(dataBuf + 7);
	offs = 9;

	if (header.version != KKE_MAP_VERSION) {
		char errmsg[64];
		sprintf(errmsg, "Expected map version %d, file is version %d.",
			KKE_MAP_VERSION, header.version);
		MessageBox(NULL, errmsg, "Failed to load map", MB_ICONEXCLAMATION | MB_OK);
		/* We'll still continue loading, even though the data may be invalid. */
	}

	/* Check if the returned values are sane. */
	if (header.numSectors > KKE_MAX_SECTORS) {
		MessageBox(NULL, "Map contains too many sectors.", "Failed to load map", MB_ICONEXCLAMATION | MB_OK);
		rc = -1;
		goto cleanup;
	}

	poMap->header = header;
	for (i = 0; i < header.numSectors; i++) {
		kke_sector_t *sc = &poMap->sectors[i];
		sc->x0 = *(unsigned short*)(dataBuf + offs);
		sc->y0 = *(unsigned short*)(dataBuf + offs + 2);
		sc->x1 = *(unsigned short*)(dataBuf + offs + 4);
		sc->y1 = *(unsigned short*)(dataBuf + offs + 6);
		sc->floorHeight = *(dataBuf + offs + 8);
		sc->ceilHeight = *(dataBuf + offs + 9);
		offs += 10;
	}

	/* TODO: Load half-edges? They will be overwritten on save anyway */
	header.numHalfEdges = 0;

	Map_SetFileName(poMap, szFilename);
	poMap->bNeedsSave = 0;

cleanup:
	CloseHandle(hFile);
	free(dataBuf);
	return rc;
}

int Map_WriteToFile(struct MapDocument *prMap, const char *szFilename)
{
	HANDLE hFile;

	DWORD dwBytesWritten;
	char *dataBuf;
	
	int sectorsSize = KKE_SECTOR_SIZE * prMap->header.numSectors;
	int edgesSize   = KKE_HALFEDGE_SIZE * prMap->header.numHalfEdges;
	int totalSize   = KKE_HEADER_SIZE + sectorsSize + edgesSize;

	int i, offs;

	offs = 0;
	dataBuf = malloc(totalSize);

	memcpy(dataBuf,     &prMap->header.version,      1);
	memcpy(dataBuf + 1, &prMap->header.startX,       2);
	memcpy(dataBuf + 3, &prMap->header.startY,       2);
	memcpy(dataBuf + 5, &prMap->header.numSectors,   2);
	memcpy(dataBuf + 7, &prMap->header.numHalfEdges, 2);
	offs += 9;

	for (i = 0; i < prMap->header.numSectors; i++) {
		const kke_sector_t *sc = &prMap->sectors[i];
		memcpy(dataBuf + offs,     &sc->x0, 2);
		memcpy(dataBuf + offs + 2, &sc->y0, 2);
		memcpy(dataBuf + offs + 4, &sc->x1, 2);
		memcpy(dataBuf + offs + 6, &sc->y1, 2);
		memcpy(dataBuf + offs + 8, &sc->floorHeight, 1);
		memcpy(dataBuf + offs + 9, &sc->ceilHeight, 1);
		offs += 10;
	}

	for (i = 0; i < prMap->header.numHalfEdges; i++) {
		memcpy(dataBuf + offs, &prMap->edges[i], KKE_HALFEDGE_SIZE);
		offs += KKE_HALFEDGE_SIZE;
	}

	/* Write the data buffer to the given filename. */
	hFile = CreateFile(szFilename, 
		GENERIC_WRITE, FILE_SHARE_WRITE, NULL,
		CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

	if (hFile == INVALID_HANDLE_VALUE)
		return -1;

	if (WriteFile(hFile, dataBuf, totalSize, &dwBytesWritten, NULL) == FALSE)
		return -1;

	Map_SetFileName(prMap, szFilename);
	prMap->bNeedsSave = 0;

	CloseHandle(hFile);
	free(dataBuf);
	return 0;
}

void Map_Destroy(const struct MapDocument *prMap)
{
	free(prMap->szFileName);
}
