#include "view.h"

#include <stdio.h>

#define GRID_SIZE 16

static HBRUSH s_hbrBackground;
static HBRUSH s_hbrStartPos;
static HPEN   s_sectorPen;
static HPEN   s_gridPen;
static HPEN   s_selectPen;

static kke_sector_t *s_pDragSector     = NULL;	// Sector currently being dragged, if any
static kke_sector_t *s_pSelectedSector = NULL;

void View_Setup()
{
	LOGBRUSH lb;
	
	s_hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	s_hbrStartPos   = CreateSolidBrush(RGB(0, 255, 0));

	lb.lbStyle  = BS_SOLID;
	lb.lbColor  = RGB(255, 0, 0);
	s_sectorPen = ExtCreatePen(PS_COSMETIC | PS_SOLID, 1, &lb, 0, NULL);

	lb.lbColor = RGB(0, 0, 127);
	s_gridPen  = ExtCreatePen(PS_COSMETIC | PS_SOLID, 1, &lb, 0, NULL);

	lb.lbColor = RGB(255, 255, 255);
	s_selectPen = ExtCreatePen(PS_COSMETIC | PS_DOT, 1, &lb, 0, NULL);
}

void View_StartDrag(POINTS point, struct MapDocument *prMap)
{
	kke_sector_t sc;
	sc.x0 = point.x / GRID_SIZE;
	sc.y0 = point.y / GRID_SIZE;
	sc.x1 = sc.x0 + 1;
	sc.y1 = sc.y0 + 1;
	sc.floorHeight = 128;
	sc.ceilHeight  = 132;

	if (Map_IntersectsWithAnother(prMap, &sc)) {
		MessageBeep(MB_OK);
		return;
	}
	
	s_pDragSector = Map_AddSector(prMap, sc);
}

void View_ContinueDrag(POINTS dragStart, POINTS dragEnd, const struct MapDocument *prMap)
{
	u16 oldX1, oldY1;

	if (s_pDragSector == NULL)
		return;

	oldX1 = s_pDragSector->x1;
	oldY1 = s_pDragSector->y1;

	s_pDragSector->x1 = s_pDragSector->x0 + (dragEnd.x - dragStart.x) / GRID_SIZE + 1;
	s_pDragSector->y1 = s_pDragSector->y0 + (dragEnd.y - dragStart.y) / GRID_SIZE + 1;

	// We need to compare against the original pointer this function's doesn't
	// check the sector against itself.
	// i am so god damn tired
	if (Map_IntersectsWithAnother(prMap, s_pDragSector)) {
		// Don't continue if we would overlap with another sector
		s_pDragSector->x1 = oldX1;
		s_pDragSector->y1 = oldY1;
		return;
	}
}

void View_EndDrag()
{
	/* s_pSelectedSector = s_pDragSector; */
	s_pDragSector = NULL;
}

BOOL View_SelectAtPoint(POINTS point, struct MapDocument *prMap)
{
	int px = point.x / GRID_SIZE;
	int py = point.y / GRID_SIZE;
	s_pSelectedSector = Map_SectorAtPoint(prMap, px, py);
	return s_pSelectedSector != NULL;
}

BOOL View_HasSelection()
{
	return s_pSelectedSector != NULL;
}

BOOL View_GetSectorHeights(int *floor, int *ceil)
{
	if (s_pSelectedSector == NULL)
		return FALSE;
	*floor = s_pSelectedSector->floorHeight;
	*ceil  = s_pSelectedSector->ceilHeight;
	return TRUE;
}

void View_SetSectorHeights(int floor, int ceil)
{
	if (s_pSelectedSector == NULL)
		return;
	s_pSelectedSector->floorHeight = floor;
	s_pSelectedSector->ceilHeight  = ceil;
}

void View_SetStartPoint(POINTS point, struct MapDocument *prMap)
{
	prMap->header.startX = point.x / GRID_SIZE;
	prMap->header.startY = point.y / GRID_SIZE;
	prMap->bNeedsSave = 1;
}

void View_Paint(HDC hdc, LPRECT lpClientRect, const struct MapDocument *prMap)
{
	HBRUSH hBrushOld;
	HPEN   hPenOld;

	int verLines, horizLines;
	int i;

	/* Fill background colour. */
	hBrushOld = SelectObject(hdc, s_hbrBackground);
	FillRect(hdc, lpClientRect, s_hbrBackground);

	/* Draw a grid across the background. */
	hPenOld = SelectObject(hdc, s_gridPen);

	verLines   = (lpClientRect->right - lpClientRect->left) / GRID_SIZE + 1;
	horizLines = (lpClientRect->bottom - lpClientRect->top) / GRID_SIZE + 1;

	for (i = 0; i < verLines; i++) {
		int xpos = lpClientRect->left + i * GRID_SIZE;
		MoveToEx(hdc, xpos, lpClientRect->top, NULL);
		LineTo(hdc, xpos, lpClientRect->bottom);
	}

	for (i = 0; i < horizLines; i++) {
		int ypos = lpClientRect->top + i * GRID_SIZE;
		MoveToEx(hdc, lpClientRect->left, ypos, NULL);
		LineTo(hdc, lpClientRect->right, ypos);
	}

	/* Draw a rectangle for each sector. */
	SelectObject(hdc, s_sectorPen);
	for (i = 0; i < prMap->header.numSectors; i++) {
		const kke_sector_t *sc = &prMap->sectors[i];
		int x0 = sc->x0 * GRID_SIZE;
		int y0 = sc->y0 * GRID_SIZE;
		int x1 = sc->x1 * GRID_SIZE;
		int y1 = sc->y1 * GRID_SIZE;

		Rectangle(hdc, x0, y0, x1, y1);

	}

	/* Draw starting position of player. */
	SelectObject(hdc, s_hbrStartPos);
	{
		int x0 = prMap->header.startX * GRID_SIZE;
		int y0 = prMap->header.startY * GRID_SIZE;
		int x1 = (prMap->header.startX + 1) * GRID_SIZE;
		int y1 = (prMap->header.startY + 1) * GRID_SIZE;
		Ellipse(hdc, x0, y0, x1, y1);
	}

	if (s_pSelectedSector != NULL) {
		int x0 = s_pSelectedSector->x0 * GRID_SIZE;
		int y0 = s_pSelectedSector->y0 * GRID_SIZE;
		int x1 = s_pSelectedSector->x1 * GRID_SIZE;
		int y1 = s_pSelectedSector->y1 * GRID_SIZE;

		SelectObject(hdc, GetStockObject(NULL_BRUSH));
		SelectObject(hdc, s_selectPen);
		Rectangle(hdc, x0 - 1, y0 - 1, x1 + 1, y1 + 1);
	}

	/* Restore previous HDC settings. */
	SelectObject(hdc, hBrushOld);
	SelectObject(hdc, hPenOld);
}

void View_Destroy()
{
	DeleteObject(s_hbrBackground);
	DeleteObject(s_sectorPen);
}
